function [T,z] = fault_slip_cheb_matrix_T(N,M,points_type,z_inp)
%
% Computes an M x N matrix with N first-order Chebyshev polynomials als columns, discretized 
% at M points z. Uses a recursive formulation.
%
% The input variable "points_type" determines the location of the points z:
%   1 = zeros of T_n(z) where -1 < z < 1 (regular first-kind Chebyshev points)
%   2 = zeros of U_n(z) where -1 < z < 1 (regular second-kind Chebyshev points)
%   3 = zeros of sqrt(1-z^2) U_(n-2)(z) where -1 <= z <= 1 (modified second-kind Cheb. points)
%   4 = equidistantly distributed excluding the end points: -1 < z < 1 
%   5 = equidistantly distributed including the end points: -1 <= z <= 1 
%   6 = specified as z_inp through header
%
% Reference: Mason & Handscomb (2003), Ch 1. 
%

switch points_type
    case 1
        z = -cos( pi*((1:M)-0.5)/M );
    case 2
        z = -cos( pi*(1:M)/(M+1) );
    case 3
        z = -cos( pi*(0:M-1)/(M-1) );
    case 4
        help = linspace(-1,1,2*M+1);
        z = help(2:2:end-1); % half spacing at beginning and end
    case 5
        z = linspace(-1,1,M);
    case 6
        z = z_inp;
        if length(z) ~= M
            error('\rDiscrepancy between length(z) and M in fault_slip_cheb_matrix_T.') 
        end
end
T = zeros(M,N);
T(:,1) = 1; % T_0 % Chebyshev polynomials start counting at 0
T(:,2) = z'; % T_1
for i = 3:N
    T(:,i) = 2*z'.*T(:,i-1)-T(:,i-2); % T_2 ... T_{N-1}
end
